<?php
session_start();
require_once 'config/database.php';

// Layout variables
$page_title = 'DMAppsESP - Sistem Bilgileri';
$breadcrumb_title = 'Sistem Bilgileri';
$current_page = 'system';
$extra_css = '<style>
    .info-card {
        background: white;
        border-radius: 10px;
        padding: 20px;
        margin-bottom: 20px;
        box-shadow: 0 2px 10px rgba(0,0,0,0.1);
    }
    .metric-item {
        display: flex;
        justify-content: space-between;
        align-items: center;
        padding: 10px 0;
        border-bottom: 1px solid #eee;
    }
    .metric-item:last-child {
        border-bottom: none;
    }
    .metric-value {
        font-weight: bold;
        color: #007bff;
    }
    .status-good { color: #28a745; }
    .status-warning { color: #ffc107; }
    .status-danger { color: #dc3545; }
    .progress-bar {
        height: 20px;
        background-color: #e9ecef;
        border-radius: 10px;
        overflow: hidden;
    }
    .progress-fill {
        height: 100%;
        background-color: #007bff;
        transition: width 0.3s ease;
    }
    .log-item {
        font-family: monospace;
        font-size: 12px;
        padding: 5px;
        border-left: 3px solid #007bff;
        margin-bottom: 5px;
        background: #f8f9fa;
    }
    .log-error { border-left-color: #dc3545; background: #f8d7da; }
    .log-warning { border-left-color: #ffc107; background: #fff3cd; }
    .log-info { border-left-color: #17a2b8; background: #d1ecf1; }
</style>';

$extra_js = '<script>
function refreshSystemInfo() {
    fetch(\'api/system.php?action=info\')
        .then(response => response.json())
        .then(data => {
            updateSystemInfo(data);
        });
}

function updateSystemInfo(data) {
    // Sistem metrikleri
    document.getElementById(\'cpuUsage\').textContent = data.cpu_usage + \'%\';
    document.getElementById(\'memoryUsage\').textContent = data.memory_usage + \'%\';
    document.getElementById(\'diskUsage\').textContent = data.disk_usage + \'%\';
    document.getElementById(\'uptime\').textContent = data.uptime;
    
    // Progress barları güncelle
    updateProgressBar(\'cpuBar\', data.cpu_usage);
    updateProgressBar(\'memoryBar\', data.memory_usage);
    updateProgressBar(\'diskBar\', data.disk_usage);
    
    // Durum göstergeleri
    updateStatusIndicator(\'cpuStatus\', data.cpu_usage);
    updateStatusIndicator(\'memoryStatus\', data.memory_usage);
    updateStatusIndicator(\'diskStatus\', data.disk_usage);
}

function updateProgressBar(id, value) {
    const bar = document.getElementById(id);
    const fill = bar.querySelector(\'.progress-fill\');
    fill.style.width = value + \'%\';
    
    // Renkleri ayarla
    if (value > 80) {
        fill.style.backgroundColor = \'#dc3545\';
    } else if (value > 60) {
        fill.style.backgroundColor = \'#ffc107\';
    } else {
        fill.style.backgroundColor = \'#28a745\';
    }
}

function updateStatusIndicator(id, value) {
    const indicator = document.getElementById(id);
    indicator.className = \'metric-value \';
    
    if (value > 80) {
        indicator.classList.add(\'status-danger\');
    } else if (value > 60) {
        indicator.classList.add(\'status-warning\');
    } else {
        indicator.classList.add(\'status-good\');
    }
}

function refreshLogs() {
    const level = document.getElementById(\'logLevel\').value;
    const limit = document.getElementById(\'logLimit\').value;
    
    fetch(\'api/system.php?action=logs&level=\' + level + \'&limit=\' + limit)
        .then(response => response.json())
        .then(data => {
            updateLogs(data.logs);
        });
}

function updateLogs(logs) {
    const container = document.getElementById(\'logsContainer\');
    let html = \'\';
    
    if (logs && logs.length > 0) {
        logs.forEach(log => {
            const logClass = getLogClass(log.level);
            html += \'<div class="log-item \' + logClass + \'">\';
            html += \'<strong>[\' + log.timestamp + \']</strong> \';
            html += \'<span class="text-muted">\' + log.level.toUpperCase() + \'</span> - \';
            html += log.message;
            html += \'</div>\';
        });
    } else {
        html = \'<p class="text-muted">Log bulunamadı.</p>\';
    }
    
    container.innerHTML = html;
}

function getLogClass(level) {
    switch(level) {
        case \'ERROR\': return \'log-error\';
        case \'WARNING\': return \'log-warning\';
        case \'INFO\': return \'log-info\';
        default: return \'\';
    }
}

function clearLogs() {
    if (confirm(\'Tüm logları temizlemek istediğinizden emin misiniz?\')) {
        fetch(\'api/system.php?action=clear_logs\', {
            method: \'POST\'
        })
        .then(response => response.json())
        .then(data => {
            if (data.success) {
                alert(\'Loglar temizlendi!\');
                refreshLogs();
            } else {
                alert(\'Hata: \' + data.message);
            }
        });
    }
}

function runMaintenance() {
    const task = document.getElementById(\'maintenanceTask\').value;
    
    fetch(\'api/system.php?action=maintenance\', {
        method: \'POST\',
        headers: {
            \'Content-Type\': \'application/json\'
        },
        body: JSON.stringify({task: task})
    })
    .then(response => response.json())
    .then(data => {
        if (data.success) {
            alert(\'Bakım işlemi tamamlandı!\');
            refreshSystemInfo();
        } else {
            alert(\'Hata: \' + data.message);
        }
    });
}

// Otomatik yenileme
let autoRefresh = false;
let refreshInterval;

function toggleAutoRefresh() {
    autoRefresh = !autoRefresh;
    const btn = document.getElementById(\'autoRefreshBtn\');
    
    if (autoRefresh) {
        btn.textContent = \'Otomatik Yenilemeyi Durdur\';
        btn.classList.remove(\'btn-outline-primary\');
        btn.classList.add(\'btn-primary\');
        
        refreshInterval = setInterval(refreshSystemInfo, 5000);
        refreshSystemInfo();
    } else {
        btn.textContent = \'Otomatik Yenilemeyi Başlat\';
        btn.classList.remove(\'btn-primary\');
        btn.classList.add(\'btn-outline-primary\');
        
        clearInterval(refreshInterval);
    }
}

// Sayfa yüklendiğinde bilgileri getir
document.addEventListener(\'DOMContentLoaded\', function() {
    refreshSystemInfo();
    refreshLogs();
});
</script>';

require_once 'layouts/header.php';
?>

<div class="container-fluid">
    <div class="row">
        <div class="col-12">
            <h1 class="h3 mb-4">Sistem Bilgileri</h1>
            
            <!-- Sistem Durumu -->
            <div class="info-card">
                <div class="d-flex justify-content-between align-items-center mb-3">
                    <h5>Sistem Durumu</h5>
                    <button class="btn btn-outline-primary btn-sm" onclick="toggleAutoRefresh()" id="autoRefreshBtn">Otomatik Yenilemeyi Başlat</button>
                </div>
                
                <div class="row">
                    <div class="col-md-4">
                        <div class="metric-item">
                            <span>CPU Kullanımı:</span>
                            <span class="metric-value" id="cpuStatus">0%</span>
                        </div>
                        <div class="progress-bar" id="cpuBar">
                            <div class="progress-fill" style="width: 0%"></div>
                        </div>
                    </div>
                    <div class="col-md-4">
                        <div class="metric-item">
                            <span>Bellek Kullanımı:</span>
                            <span class="metric-value" id="memoryStatus">0%</span>
                        </div>
                        <div class="progress-bar" id="memoryBar">
                            <div class="progress-fill" style="width: 0%"></div>
                        </div>
                    </div>
                    <div class="col-md-4">
                        <div class="metric-item">
                            <span>Disk Kullanımı:</span>
                            <span class="metric-value" id="diskStatus">0%</span>
                        </div>
                        <div class="progress-bar" id="diskBar">
                            <div class="progress-fill" style="width: 0%"></div>
                        </div>
                    </div>
                </div>
                
                <div class="row mt-3">
                    <div class="col-md-6">
                        <div class="metric-item">
                            <span>Sistem Çalışma Süresi:</span>
                            <span class="metric-value" id="uptime">-</span>
                        </div>
                    </div>
                    <div class="col-md-6">
                        <div class="metric-item">
                            <span>Sistem Yükü:</span>
                            <span class="metric-value">-</span>
                        </div>
                    </div>
                </div>
            </div>
            
            <!-- Veritabanı Bilgileri -->
            <div class="info-card">
                <h5>Veritabanı Bilgileri</h5>
                <div class="row">
                    <div class="col-md-3">
                        <div class="metric-item">
                            <span>Kullanıcılar:</span>
                            <span class="metric-value">-</span>
                        </div>
                    </div>
                    <div class="col-md-3">
                        <div class="metric-item">
                            <span>Cihazlar:</span>
                            <span class="metric-value">-</span>
                        </div>
                    </div>
                    <div class="col-md-3">
                        <div class="metric-item">
                            <span>Sensör Verileri:</span>
                            <span class="metric-value">-</span>
                        </div>
                    </div>
                    <div class="col-md-3">
                        <div class="metric-item">
                            <span>Sistem Logları:</span>
                            <span class="metric-value">-</span>
                        </div>
                    </div>
                </div>
            </div>
            
            <!-- Sistem Logları -->
            <div class="info-card">
                <div class="d-flex justify-content-between align-items-center mb-3">
                    <h5>Sistem Logları</h5>
                    <button class="btn btn-danger btn-sm" onclick="clearLogs()">Logları Temizle</button>
                </div>
                
                <div class="row mb-3">
                    <div class="col-md-3">
                        <label class="form-label">Log Seviyesi:</label>
                        <select class="form-select" id="logLevel" onchange="refreshLogs()">
                            <option value="all">Tümü</option>
                            <option value="ERROR">Hata</option>
                            <option value="WARNING">Uyarı</option>
                            <option value="INFO">Bilgi</option>
                        </select>
                    </div>
                    <div class="col-md-3">
                        <label class="form-label">Limit:</label>
                        <select class="form-select" id="logLimit" onchange="refreshLogs()">
                            <option value="50">50</option>
                            <option value="100">100</option>
                            <option value="200">200</option>
                            <option value="500">500</option>
                        </select>
                    </div>
                    <div class="col-md-6">
                        <button class="btn btn-primary" onclick="refreshLogs()">Yenile</button>
                    </div>
                </div>
                
                <div id="logsContainer" style="max-height: 300px; overflow-y: auto;">
                    <p class="text-muted">Loglar yükleniyor...</p>
                </div>
            </div>
            
            <!-- Bakım İşlemleri -->
            <div class="info-card">
                <h5>Bakım İşlemleri</h5>
                <div class="row">
                    <div class="col-md-4">
                        <label class="form-label">Bakım Görevi:</label>
                        <select class="form-select" id="maintenanceTask">
                            <option value="optimize_database">Veritabanı Optimize Et</option>
                            <option value="clear_cache">Önbelleği Temizle</option>
                            <option value="cleanup_old_data">Eski Verileri Temizle</option>
                            <option value="rebuild_indexes">Indexleri Yeniden Oluştur</option>
                            <option value="backup_database">Veritabanı Yedekle</option>
                        </select>
                    </div>
                    <div class="col-md-4">
                        <label class="form-label">&nbsp;</label>
                        <button class="btn btn-warning" onclick="runMaintenance()">Çalıştır</button>
                    </div>
                </div>
            </div>
            
            <!-- PHP Bilgileri -->
            <div class="info-card">
                <h5>PHP Bilgileri</h5>
                <div class="row">
                    <div class="col-md-3">
                        <div class="metric-item">
                            <span>PHP Versiyonu:</span>
                            <span class="metric-value"><?php echo PHP_VERSION; ?></span>
                        </div>
                    </div>
                    <div class="col-md-3">
                        <div class="metric-item">
                            <span>Memory Limit:</span>
                            <span class="metric-value"><?php echo ini_get('memory_limit'); ?></span>
                        </div>
                    </div>
                    <div class="col-md-3">
                        <div class="metric-item">
                            <span>Max Execution Time:</span>
                            <span class="metric-value"><?php echo ini_get('max_execution_time'); ?>s</span>
                        </div>
                    </div>
                    <div class="col-md-3">
                        <div class="metric-item">
                            <span>Upload Max Filesize:</span>
                            <span class="metric-value"><?php echo ini_get('upload_max_filesize'); ?></span>
                        </div>
                    </div>
                </div>
            </div>
            
            <!-- Sunucu Bilgileri -->
            <div class="info-card">
                <h5>Sunucu Bilgileri</h5>
                <div class="row">
                    <div class="col-md-3">
                        <div class="metric-item">
                            <span>İşletim Sistemi:</span>
                            <span class="metric-value"><?php echo php_uname('s'); ?></span>
                        </div>
                    </div>
                    <div class="col-md-3">
                        <div class="metric-item">
                            <span>Sunucu Adı:</span>
                            <span class="metric-value"><?php echo $_SERVER['SERVER_NAME']; ?></span>
                        </div>
                    </div>
                    <div class="col-md-3">
                        <div class="metric-item">
                            <span>IP Adresi:</span>
                            <span class="metric-value"><?php echo $_SERVER['SERVER_ADDR']; ?></span>
                        </div>
                    </div>
                    <div class="col-md-3">
                        <div class="metric-item">
                            <span>Yazılım:</span>
                            <span class="metric-value"><?php echo $_SERVER['SERVER_SOFTWARE']; ?></span>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </div>
</div>

<?php require_once 'layouts/footer.php'; ?>
